//
//  CounterFeature.swift
//  Learn Composable Architecture
//
//  Created by Takeshi Iwana on 9/15/25.
//

import ComposableArchitecture
import SwiftUI

enum CancelId: Hashable, Sendable { case timer(UUID) }

@Reducer
struct CounterFeature {
  @ObservableState
  struct State {
    var count = 0
    var fact: String?
    var isLoading = false
    var isTimerRunning = false
    let timerID = UUID()
  }

  enum Action {
    case decrementButtonTapped
    case incrementButtonTapped
    case factButtonTapped
    case factResponse(String)
    case toggleTimerButtonTapped
    case timerTick
  }

  var body: some ReducerOf<Self> {
    Reduce { state, action in
      switch action {
      case .decrementButtonTapped:
        state.count -= 1
        state.fact = nil
        return .none
      case .incrementButtonTapped:
        state.count += 1
        state.fact = nil
        return .none
      case .factButtonTapped:
        state.fact = nil
        state.isLoading = true
        return .run { [count = state.count] send in
          let (data, _) = try await URLSession.shared
            .data(from: URL(string: "http://numbersapi.com/\(count)")!)
          let fact = String(decoding: data, as: UTF8.self)
          await send(.factResponse(fact))
        }
      case let .factResponse(fact):
        state.fact = fact
        state.isLoading = false
        return .none
      case .toggleTimerButtonTapped:
        state.isTimerRunning.toggle()
          let cancelID = CancelId.timer(state.timerID)
          if state.isTimerRunning {
          return .run { send in
            while true {
              try await Task.sleep(for: .seconds(1))
              await send(.timerTick)
            }
          }.cancellable(id:cancelID)
        } else {
            return .cancel(id: cancelID)
        }
      case .timerTick:
        state.count += 1
        state.fact = nil
        return .none

      }
    }
  }
}

struct CounterView: View {
  let store: StoreOf<CounterFeature>
  var body: some View {
    VStack {
      Text("\(store.count)")
        .font(.largeTitle)
        .padding()
        .background(Color.black.opacity(0.1))
        .cornerRadius(10)
      HStack {
        Button("-") {
          store.send(.decrementButtonTapped)
        }
        .font(.largeTitle)
        .padding()
        .background(Color.red.opacity(0.1))
        .cornerRadius(10)

        Button("+") {
          store.send(.incrementButtonTapped)
        }
        .font(.largeTitle)
        .padding()
        .background(Color.blue.opacity(0.1))
        .cornerRadius(10)
      }
      Button(store.isTimerRunning ? "Stop" : "Start") {
        store.send(.toggleTimerButtonTapped)
      }
      Button("Fact") {
        store.send(.factButtonTapped)
      }
      .font(.largeTitle)
      .padding()
      .background(Color.black.opacity(0.1))
      .cornerRadius(10)

      if store.isLoading {
        ProgressView()
      } else if let fact = store.fact {
        Text(fact)
          .font(.largeTitle)
          .multilineTextAlignment(.center)
          .padding()
      }
    }
  }
}

#Preview{
  CounterView(
    store: Store(initialState: CounterFeature.State()) {
      CounterFeature()
    }
  )
}
